#!/usr/bin/env python
'''
make dom0 dump file from xen whole machine dump file
'''


import SymbolTable
import PageTable
import debug_info
import os.path
import array
from optparse import OptionParser
import XenCore
import ElfCore


typemap = {
    'xen': XenCore.XenCoreReader,
    'elf': ElfCore.ElfCoreReader,
    }


class XenImage:
    '''
    Xen address space and symbols environment to analyze dump file
    '''

    def __init__(self, dump, xensyms):
        self.dump = dump
        # symbol table
        self.st = SymbolTable.SymbolTable()
        self.st.loadSymbolFromObjectFile(xensyms)

        self.pt = self.get_xen_idle_pt()

        # DWARF debug information tree
        self.tree = debug_info.TagTree()
        self.tree.buildTree(xensyms)

    def get_xen_idle_pt(self):
        '''idle domain page table'''
        
        ptaddr = self.st['idle_pg_table'] - 0xff000000L # PAGE_OFFSET 
        return PageTable.PageTable(self.dump, self.dump.arch.maddr_to_mfn(ptaddr))

    def get_dom0p(self):
        """domain.arch_domain.mm_perdomain_pt 's offset """
        
        return self.dump.read_struct(self.pt.v2m(self.st['dom0']), 'L')[0]

    def get_domain_pt(self, domainp):
        '''get kernel pagetable for domain. domainp is virtual address for struct domain'''
        
        vcpu_offset = self.tree.calc_struct_member_offset('domain', 'vcpu') # vcpu[0]
        table_offset = self.tree.calc_struct_member_offset('vcpu', 'arch')
        table_offset += self.tree.calc_struct_member_offset('arch_vcpu', 'guest_table')

        # domain->vcpu[0]->arch.guest_table
        domain_vcpu = self.dump.read_struct(self.pt.v2m(domainp + vcpu_offset), 'L')[0]
        domain_pt_mfn = self.dump.read_struct(self.pt.v2m(domain_vcpu + table_offset), 'L')[0]

        # domain pagetable
        return PageTable.PageTable(self.dump, domain_pt_mfn)

    def get_domain_context(self, domainp):
        '''get domain context informations'''
        
        # XXX UniProcessor only!! need SMP
        # domain->vcpu[0]
        vcpu_offset = self.tree.calc_struct_member_offset('domain', 'vcpu') # vcpu[0]
        domain_vcpu = self.dump.read_struct(self.pt.v2m(domainp + vcpu_offset), 'L')[0]

        # vcpu->arch.vcpu_guest_context
        context_offset = self.tree.calc_struct_member_offset('vcpu', 'arch')
        context_offset += self.tree.calc_struct_member_offset('arch_vcpu', 'guest_context')
        context_size = self.tree.get_struct_size('vcpu_guest_context')

        return self.pt.load_data(domain_vcpu + context_offset, context_size)


class DomainImage:
    '''
    Domain address space and symbols environment to analyze dump file
    '''

    def __init__(self, dump, vmlinux, domain_pt):
        self.dump = dump
        self.pt = domain_pt

        # symbol table
        self.st = SymbolTable.SymbolTable()
        self.st.loadSymbolFromObjectFile(vmlinux)

        ## DWARF debug information tree isn't used, yet
        #self.tree = debug_info.TagTree()
        #self.tree.buildTree(vmlinux)

        self.max_pfn = self.get_max_pfn()
        self.pfn2mfn = self.get_p2m()
        self.nr_pages = len(self.pfn2mfn)

    def get_max_pfn(self):
        '''return max page frame number'''

        # read data
        return self.dump.read_struct(self.pt.v2m(self.st['max_pfn']), 'L')[0]

    def get_p2m(self):
        '''return pfn2mfn mapping dict'''

        p2m_table = self.dump.read_struct(self.pt.v2m(self.st['phys_to_machine_mapping']), 'L')[0]

        # make pfn2mfn from phys_to_machine_mapping
        p2m = array.array('L')
        p2m.fromstring(self.pt.load_data(p2m_table, self.max_pfn * 4))
        pfn2mfn = {}
        for pfn, mfn in enumerate(p2m):
            if mfn != 0xffffffffL:
                pfn2mfn[pfn] = mfn
        return pfn2mfn

    def need_pages(self):
        '''return need pages by list of mfn'''
        pages = self.pfn2mfn.values()
        pages.sort()
        return pages


def dom0extract(outdump, dump, xensyms, vmlinux):
    '''analyxe dump with xen-syms, vmlinux, arch
    to make dom0 dump file outdump.
    '''

    xenimg = XenImage(dump, xensyms)

    # get dom0 info
    dom0p = xenimg.get_dom0p()
    dom0_pt = xenimg.get_domain_pt(dom0p)
    dom0context = xenimg.get_domain_context(dom0p)

    # get dom0 image
    dom0img = DomainImage(dump, vmlinux, dom0_pt)
    
    # set header info
    outdump.set_context(1, dom0context)
    outdump.set_nr_pages(dom0img.nr_pages)

    pages = dom0img.need_pages()
    outdump.fetch_pages(dump, pages)

    # write header
    outdump.writeheader()


def main():
    '''command line interface function'''

    oparser = OptionParser()
    oparser.add_option('-o', '--dom0dump', dest='dom0dumpname', default='dom0mem', type='string', help='dom0 dump file for output')
    oparser.add_option('-d', '--xendump', dest='dumpname', default='dump', type='string', help='xen dump file for input')
    oparser.add_option('-x', '--xensyms', dest='xensyms', default='xen-syms', type='string', help='xen binary with symbols')
    oparser.add_option('-v', '--vmlinux', dest='vmlinux', default='vmlinux', type='string', help='linux binary with symbols')
    oparser.add_option('-a', '--arch', dest='arch', type='string', default='x86_32', help="architecture (now 'x86_32' only)")
    oparser.add_option("-t", "--type", dest="filetype", default="xen", type="string", help='dump file type("xen" or "elf")')

    (options, args) = oparser.parse_args()

    
    if not os.path.exists(options.dumpname):
        print options.dumpname, 'doesn\'t exists!'
        return
    if not os.path.exists(options.xensyms):
        print options.xensyms, 'doesn\'t exists!'
        return
    if not os.path.exists(options.vmlinux):
        print options.vmlinux, 'doesn\'t exists!'
        return
    if options.arch != 'x86_32':
        print 'not supported architecture: ', options.arch
        return
    
    # filetype, file, arch
    coretype = typemap[options.filetype]
    dump = coretype(options.dumpname, options.arch)

    if os.path.exists(options.dom0dumpname):
        print options.dom0dumpname, 'already exists!'
        return
    else:
        file(options.dom0dumpname, 'w')      # create blank file

    dom0dump = XenCore.XenCoreWriter(options.dom0dumpname, options.arch)
    dom0extract(dom0dump, dump, options.xensyms, options.vmlinux)

if __name__ == '__main__':
    main()

