import os
import re
import CoreDump

command = 'LANG=C readelf -l %s' 

class ElfSection:
    '''
    Elf Section header class.
    '''

    def __init__(self, typename, offset, vaddr, maddr, filesize, memsize, flag, align):
        self.typename = typename
        self.offset = offset
        self.vaddr = vaddr
        self.maddr = maddr
        self.filesize = filesize
        self.memsize = memsize
        self.flag = flag
        self.align = align

    def has_maddr(self, maddr):
        return self.maddr <= maddr < self.maddr + self.memsize 
        
    def maddr_to_offset(self, maddr):
        return maddr - self.maddr + self.offset

class ElfCoreReader(CoreDump.CoreDump):
    '''
    Elf Core file reader interface
    '''

    def __init__(self, corefilename, arch):
        CoreDump.CoreDump.__init__(self, corefilename, arch)
        self.sections = self._read_sections()
        self.file = file(corefilename, 'r')
        if len(self.sections) == 0:
            raise ValueError, 'section not found in ELF format'
    
    def _read_sections(self):
        '''
        read section header informations via readelf
        '''

        sections = []
        f = os.popen(command % self.corefilename, 'r')

        section_pattern = '\s+(?P<typename>\w+)\s+(?P<offset>0x[0-9a-f]+) (?P<vaddr>0x[0-9a-f]+) (?P<paddr>0x[0-9a-f]+) (?P<filesize>0x[0-9a-f]+) (?P<memsize>0x[0-9a-f]+) (?P<flag>[R ][W ][E ]) (?P<align>[0-9]+)'
        section_prog = re.compile(section_pattern)

        for line in f:
            m = section_prog.search(line)
            if m:
                (typename, offset, vaddr, paddr, filesize, memsize, flag, align) = m.groups()

                offset = int(offset, 16)
                paddr = int(paddr, 16)
                filesize = int(filesize, 16)
                memsize = int(memsize, 16)
                align = int(align, 16)
                
                sections.append( ElfSection(typename, offset, vaddr, paddr, filesize, memsize, flag, align) )

        return sections

    def read_page(self, mfn):
        '''return a page data'''

        offset = None
        for sec in self.sections:
            if sec.has_maddr(self.arch.mfn_to_maddr(mfn)):
                offset = sec.maddr_to_offset(self.arch.mfn_to_maddr(mfn))
                break
        if offset == None:
            raise KeyError, '%s not found' % mfn
        self.file.seek(offset)
        data = self.file.read(self.arch.page_size)
        return data
    
    def has_page(self, mfn):
        '''return a page is there or not'''

        for sec in self.sections:
            if sec.has_maddr(self.arch.mfn_to_maddr(mfn)):
                return True
        return False

    def get_pagelist(self):
        '''return a list of all available mfn'''
        
        pages = []
        for sec in self.sections:
            start_mfn = self.arch.maddr_to_mfn(sec.maddr)
            end_mfn = self.arch.maddr_to_mfn(sec.maddr + sec.memsize)
            pages.append(range(start_mfn, end_mfn+1))
        return pages
            
            
