import _libelf
from libelf import *
from exceptions import Exception

class ElfException(Exception):
    '''Exception detected by libelf'''
    pass

def begin(format, ver, objfile, cmd, ref):
    '''create or duplicate Elf object from file'''

    ret = elf_version(ver)
    if ret == EV_NONE:
        raise ElfException(elf_errmsg(elf_errno()))

    try:
        if format == 'ELF32':
            return Elf32(elf_begin(objfile.fileno(), cmd, ref))
        elif format == 'ELF64':
            return Elf64(elf_begin(objfile.fileno(), cmd, ref))
    except TypeError:
        raise ElfException(elf_errmsg(elf_errno()))

def memory(image, size):
    '''create Elf object from memory image'''
    try:
        return Elf(elf_memory(image, size))
    except TypeError:
        raise ElfException(elf_errmsg(elf_errno()))

def _accessorfuncs(classname):
    def getattr(self, name):
        wrapname = '%s_%s_get' % (classname, name)
        if wrapname in _libelf.__dict__:
            return _libelf.__dict__[wrapname](self.ptr)
        elif name in self.__dict__:
            return self.__dict__[name]
        raise AttributeError
    
    def setattr(self, name, val):
        wrapname = '%s_%s_set' % (classname, name)
        if wrapname in _libelf.__dict__:
            _libelf.__dict__[wrapname](self.ptr, val)
        else:
            self.__dict__[name] = val
    return getattr, setattr


class Elf(object):
    '''ELF format file class'''
    def __init__(self, ptr):
        if ptr == None:
            raise TypeError
        self.ptr = ptr

    __getattr__, __setattr__ = _accessorfuncs('Elf')

    def end(self):
        '''close elf file object'''
        return elf_end(self.ptr)

    def cntl(self, cmd):
        if elf_cntl(self.ptr, cmd) != 0:
            raise ElfException(elf_errmsg(elf_errno()))

    def kind(self):
        '''return kind of the file (ELF_K_*) '''
        return elf_kind(self.ptr)

    def flagehdr(self, cmd, flags):
        return elf_flagehdr(self.ptr, cmd, flags)

    def flagelf(self, cmd, flags):
        return elf_flagelf(self.ptr, cmd, flags)

    def flagphdr(self, cmd, flags):
        return elf_flagphdr(self.ptr, cmd, flags)


    def getarhdr(self):
        '''get archive header'''
        try:
            return ElfArhdr(elf_getarhdr(self.ptr))
        except TypeError:
            raise ElfException(elf_errmsg(elf_errno()))

    def getarsym(self):
        '''get archive symbol table'''
        arsym, size = elf_getarsym(self.ptr)
        try:
            return ElfArsym(arsym, size)
        except TypeError:
            raise ElfException(elf_errmsg(elf_errno()))

    def getbase(self):
        '''get offset for archive member'''
        offset = elf_getbase(self.ptr)
        if offset == -1:
            raise ElfException(elf_errmsg(elf_errno()))
        return offset

    def getident(self):
        '''get identification data(e_ident)'''
        ident, size = elf_getident(self.ptr)
        if size == None:
            raise ElfException(elf_errmsg(elf_errno()))
        return ident[:size]

    def getscn(self, index):
        '''get indexed section'''
        try:
            return self.ElfScn(elf_getscn(self.ptr, index))
        except TypeError:
            raise ElfException(elf_errmsg(elf_errno()))

    def newscn(self):
        try:
            return self.ElfScn(elf_newscn(self.ptr))
        except TypeError:
            raise ElfException(elf_errmsg(elf_errno()))

    def nextscn(self, elfscn):
        '''get nextsection'''
        try:
            return self.ElfScn(elf_nextscn(self.ptr, elfscn))
        except TypeError:
            raise ElfException(elf_errmsg(elf_errno()))

    def getehdr(self):
        '''get elf header'''
        try:
            return self.ElfEhdr(self.elf_getehdr(self.ptr))
        except TypeError:
            raise ElfException(elf_errmsg(elf_errno()))

    def newehdr(self):
        '''create elf header'''
        try:
            return self.ElfEhdr(self.elf_newehdr(self.ptr))
        except TypeError:
            raise ElfException(elf_errmsg(elf_errno()))

    def getphdr(self):
        '''get program header list'''
        try:
            return self.phdrlist
        except TypeError:
            raise ElfException(elf_errmsg(elf_errno()))

    def newphdr(self, count):
        '''create program headers and return a list'''
        try:
            self.phdrarrayptr = self.elf_newphdr(self.ptr, count)
            self.phdrlist = [self.ElfPhdr(self.ElfPhdrArray_getitem(self.phdrarrayptr, i).this) for i in range(count)]
            return self.phdrlist
        except TypeError:
            raise ElfException(elf_errmsg(elf_errno()))

    def next(self):
        '''get next elf in this archive'''
        elf = elf_next(self.ptr)
        if int(elf) == ELF_C_NULL:
            raise ElfException(elf_errmsg(elf_errno()))
        return elf


    def rand(self, offset):
        off = elf_rand(self.ptr, offset) 
        if off == 0:
            raise ElfException(elf_errmsg(elf_errno()))
        return off

    def rawfile(self):
        rawfile, size = elf_rawfile(self.ptr)
        if size == None:
            raise ElfException(elf_errmsg(elf_errno()))
        return rawfile[:size]

    def strptr(self, section, offset):
        '''strptr(sectionnum, offset) -> pointer'''
        ptr = elf_strptr(self.ptr, section, offset)
        if ptr == None:
            raise ElfException(elf_errmsg(elf_errno()))
        return ptr

    def update(self, cmd):
        if self.phdrlist:
            for num, item in enumerate(self.phdrlist):
                self.ElfPhdrArray_setitem(self.phdrarrayptr, num, item.ptr)
        size = elf_update(self.ptr, cmd)
        if size == -1:
            raise ElfException(elf_errmsg(elf_errno()))
        return size

    def checksum(self):
        return self.elf_checksum(self.ptr)

    def fsize(self, count, ver):
        return self.elf_fsize(self.ptr, count, ver)


class ElfData(object):
    
    def __init__(self, ptr):
        if ptr == None:
            raise TypeError
        self.ptr = ptr

    __getattr__, __setattr__ = _accessorfuncs('Elf_Data')

    def flagdata(self, cmd, flags):
        return elf_flagdata(self.ptr, cmd, flags)

    def xlatetof(self, src, encode):
        try:
            return self.__class__(self.elf_xlatetof(self.ptr, src, encode))
        except TypeError:
            raise ElfException(elf_errmsg(elf_errno()))

    def xlatetom(self, src, encode):
        try:
            return self.__class__(self.elf_xlatetom(self.ptr, src, encode))
        except TypeError:
            raise ElfException(elf_errmsg(elf_errno()))


class ElfScn(object):
    '''ELF Section'''
    def __init__(self, ptr):
        if ptr == None:
            raise TypeError
        self.ptr = ptr

    __getattr__, __setattr__ = _accessorfuncs('Elf_Scn')

    def flagscn(self, cmd, flags):
        return elf_flagscn(self.ptr, cmd, flags)

    def flagshdr(self, cmd, flags):
        return elf_flagshdr(self.ptr, cmd, flags)


    def getdata(self, elfdata):
        try:
            return self.ElfData(elf_getdata(self.ptr, elfdata))
        except TypeError:
            raise ElfException(elf_errmsg(elf_errno()))

    def newdata(self):
        try:
            return self.ElfData(elf_newdata(self.ptr))
        except TypeError:
            raise ElfException(elf_errmsg(elf_errno()))

    def rawdata(self, elfdata):
        try:
            return self.ElfData(elf_rawdata(self.ptr, elfdata))
        except TypeError:
            raise ElfException(elf_errmsg(elf_errno()))


    def ndxscn(self):
        indx = elf_ndxscn(self.ptr)
        if indx == SHN_UNDEF:
            raise ElfException(elf_errmsg(elf_errno()))
        else:
            return indx

    def getshdr(self):
        try:
            return self.ElfShdr(self.elf_getshdr(self.ptr))
        except TypeError:
            raise ElfException(elf_errmsg(elf_errno()))

class ElfArhdr(object):
    '''archive header'''

    def __init__(self, ptr):
        if ptr == None:
            raise TypeError
        self.ptr = ptr
    
    __getattr__, __setattr__ = _accessorfuncs('Elf_Arhdr')


class ElfArsym(object):
    '''archive symbol table'''

    def __init__(self, ptr, size):
        if ptr == None:
            raise TypeError
        self.ptr = ptr
        self.size = size

    __getattr__, __setattr__ = _accessorfuncs('Elf_Arsym')

class ElfEhdr(object):
    def __init__(self, ptr):
        if ptr == None:
            raise TypeError
        self.ptr = ptr

class ElfPhdr(object):
    def __init__(self, ptr):
        if ptr == None:
            raise TypeError
        self.ptr = ptr

class ElfShdr(object):
    def __init__(self, ptr):
        if ptr == None:
            raise TypeError
        self.ptr = ptr



class Elf64Ehdr(ElfEhdr):
    '''elf header'''

    __getattr__, __setattr__ = _accessorfuncs('Elf64_Ehdr')

class Elf32Ehdr(ElfEhdr):
    '''elf header'''

    __getattr__, __setattr__ = _accessorfuncs('Elf32_Ehdr')

class Elf64Phdr(ElfPhdr):
    '''program header'''

    __getattr__, __setattr__ = _accessorfuncs('Elf64_Phdr')

class Elf32Phdr(ElfPhdr):
    '''program header'''

    __getattr__, __setattr__ = _accessorfuncs('Elf32_Phdr')

class Elf64Shdr(ElfPhdr):
    '''program header'''

    __getattr__, __setattr__ = _accessorfuncs('Elf64_Shdr')


class Elf32Shdr(ElfPhdr):
    '''program header'''
    __getattr__, __setattr__ = _accessorfuncs('Elf32_Shdr')


# errno = elf_errno()
# string =  elf_errmsg(err)
# elf_fill(fill)
# hashvalue = elf_hash(name)


class Elf32Data(ElfData):
    elf_xlatetof = elf32_xlatetof
    elf_xlatetom = elf32_xlatetom
    
class Elf64Data(ElfData):
    elf_xlatetof = elf64_xlatetof
    elf_xlatetom = elf64_xlatetom

class Elf32Scn(ElfScn):
    ElfData = Elf32Data
    ElfShdr = Elf32Shdr
    elf_getshdr = elf32_getshdr

class Elf64Scn(ElfScn):
    ElfData = Elf64Data
    ElfShdr = Elf64Shdr
    elf_getshdr = elf64_getshdr

class Elf32(Elf):
    elf_getehdr = elf32_getehdr
    elf_getphdr = elf32_getphdr
    elf_newehdr = elf32_newehdr
    elf_newphdr = elf32_newphdr
    elf_checksum = elf32_checksum
    elf_fsize = elf32_fsize
    ElfPhdr = Elf32Phdr
    ElfEhdr = Elf32Ehdr
    ElfPhdrArray_getitem = Elf32PhdrArray_getitem
    ElfPhdrArray_setitem = Elf32PhdrArray_setitem
    sizeof_ehdr = sizeof_elf32_ehdr
    sizeof_phdr = sizeof_elf32_phdr
    sizeof_nhdr = sizeof_elf32_nhdr
    sizeof_shdr = sizeof_elf32_shdr
    ElfScn = Elf32Scn
    
class Elf64(Elf):
    elf_getehdr = elf64_getehdr
    elf_getphdr = elf64_getphdr
    elf_newehdr = elf64_newehdr
    elf_newphdr = elf64_newphdr
    elf_checksum = elf64_checksum
    elf_fsize = elf64_fsize
    ElfPhdr = Elf64Phdr
    ElfEhdr = Elf64Ehdr
    ElfPhdrArray_getitem = Elf64PhdrArray_getitem
    ElfPhdrArray_setitem = Elf64PhdrArray_setitem
    sizeof_ehdr = sizeof_elf64_ehdr
    sizeof_phdr = sizeof_elf64_phdr
    sizeof_nhdr = sizeof_elf64_nhdr
    sizeof_shdr = sizeof_elf64_shdr
    ElfScn = Elf64Scn
