#!/usr/bin/env python
'''
DWARF2 debug information tree and tag class
'''

import os
import re

# XXX these patterns are highly depends on readelf output format
TAG_PATTERN = ' <(\d+)><([0-9a-z]+)>: Abbrev Number: \d+ \(DW_TAG_(.*)\)'
ATTR_PATTERN = '     DW_AT_([a-zA-Z0-9_]*)\s*: (.*)'
NAME_PATTERN = '(.*: )?(.+)\t'
OFFSET_PATTERN = 'DW_OP_plus_uconst: (\d+)'

tag_prog = re.compile(TAG_PATTERN)
attr_prog = re.compile(ATTR_PATTERN)
name_prog = re.compile(NAME_PATTERN)
offset_prog = re.compile(OFFSET_PATTERN)

command = 'LANG=C readelf -wi %s'

class TagTree:
    '''
    DWARF2 debug information tree and tag class
    '''
    def __init__(self):
        self.root = Tag()
        self.root.level = -1
        self.root.parent = self.root
        self.namemap = {}

    def buildTree(self, filename):
        '''build debug information tree from readelf -wi output
        filename: object file with debug info'''
        
        f = os.popen(command % filename, 'r')
        
        tag = self.root
        last_n = [None]

        for line in f.xreadlines():
            line = line[0:-1]

            # attribute line
            m = attr_prog.search(line)
            if m:
                (key, val) = (m.group(1), m.group(2))
                tag.add_attr(key, val)

                if key == 'name':
                    name = tag.name
                    if not self.namemap.has_key(name):
                        self.namemap[name] = [tag]
                    self.namemap[name].append(tag)
                continue

            # tag line
            m = tag_prog.search(line)
            if m:
                tag = Tag(level=int(m.group(1)), typename=m.group(3))

                level = tag.level
                if level == 0:
                    tag.parent = self.root
                else:
                    if level+1 >= len(last_n):
                        last_n.append(None)
                    tag.parent = last_n[level - 1]
                    last_n[level] = tag
                continue

    def find_struct_member(self, structname, membername):
        '''return list of member tags for struct.member by name'''
        structs = self.find(Tag.is_struct, structname)
        # XXX return only 1st structure
        for tag in structs:
            members = self.find(Tag.is_member, membername, parent=tag)
            if members:
                return members

    def find(self, func, name, parent=None):
        '''find tags by name, callback func, and parent tag.
        parent node setting is optional'''

        if self.namemap.has_key(name):
            tags = self.namemap[name]
        else:
            return []
        tags = [tag for tag in tags if func(tag)]
        if parent:
            tags = [tag for tag in tags if tag.is_child_of(parent)]
        return tags
        
    def calc_struct_member_offset(self, structname, membername):
        '''return member offset from structure'''

        member = self.find_struct_member(structname, membername)
        return member[0].get_offset()

    def get_struct_size(self, structname):
        '''return structure size in bytes'''

        structs = self.find(Tag.is_struct, structname)
        return structs[0].size

class Tag:
    '''
    DWARF2 debug information tag.
    It should be used in TagTree
    '''

    def __init__(self, level=None, typename=None):
        self.level = level
        self.typename = typename
        self.attr = {}
        self.offset = None
        self.name = None
        self.parent = None

    def add_attr(self, key, val):
        '''add member of attributes.
        attribute name is key,
        attribute value is val.
        '''
        if key == 'name':               
            m = name_prog.search(val)
            if m:
                self.name = m.group(2)
            else:
                print key, val
            return

        if key == 'data_member_location':
            m = offset_prog.search(val)
            if m:
                self.offset = int(m.group(1))
            else:
                print key, val
            return

        if key == 'byte_size':
            self.size = int(val)

    def is_struct(self):
        '''this tag is struct?'''
        return self.typename == 'structure_type'

    def is_member(self):
        '''this tag is member?'''
        return self.typename == 'member'

    def is_child_of(self, other):
        '''this tag is child of other?'''
        return other == self.parent

    def is_name(self, name):
        '''this tag have this name?'''
        return self.name == name

    def get_offset(self):
        '''return offset value'''
        return self.offset
