/**************************************************************************\
*//*! \file ef_bend.h Xen back end driver main data structure

Copyright 2006 Solarflare Communications Inc,
               9501 Jeronimo Road, Suite 250,
               Irvine, CA 92618, USA

This program is free software; you can redistribute it and/or modify it under
the terms of the GNU General Public License version 2 as published by the Free
Software Foundation, incorporated herein by reference.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

 *//*
\**************************************************************************/
#ifndef EF_BEND_H
#define EF_BEND_H

#include "ci/xen/ef_shared_fifo.h"
#include "ci/xen/ef_msg_iface.h"

#ifdef BEND_VERBOSE
#define BEND_VERB(_x) _x
#else
#define BEND_VERB(_x) do {} while(0)
#endif

/*! We talk version 0.010 of the interdomain protocol */
#define EF_VPROTO_VERSION (0x00001000)

#define MAC_FMT "%.2x:%.2x:%.2x:%.2x:%.2x:%.2x"
#define MAC_ARG(_mac) (_mac)[0], (_mac)[1], (_mac)[2], (_mac)[3], (_mac)[4], (_mac)[5]

/*! Resource limits for a given NIC */
struct ef_bend_limits {
  int max_filters;                    /*!< Max. number of filters to use. */
  int max_mcasts;                     /*!< Max. number  of mcast subscriptions */
  int max_buf_pages;                  /*!< Max. number of pages of NIC buffers */
  int max_tx_rate;                    /*!< Max. data rate (Mbps)(if available) */
};

/*! The state for an instance of the back end driver. */
struct ef_bend {
  struct ef_hyperop_info hinfo;       /*! Hypervisor-specific state */
  /* Structures for communication with client VM that are assumed to be
  * implemented for any hypervisor */
  void *hdev_data;                    /*!< Pointer to whatever device cookie ties
                                       * us in to the hypervisor, e.g. the 
                                       * xenbus_device on XEN */
  ef_notify_t channel;                /*!< General notification */
  struct ef_shared_page *shared_page; /*!< FIFO indices. Next page is msg FIFOs */
  struct work_struct state_change;    /*!< Defer Xenbus state changes out of IRQ 
                                        * as they can sleep. */
  struct work_struct handle_msg;      /*!< Defer control message processing */
  int newt_state;                     /*!< State to change to. */
  ef_vmident_t far_end;               /*!< Identifies other end VM .*/
  struct ef_hyperop_mapping *sh_pages_unmap;  /*!< To unmap the shared pages */
  struct ef_hyperop_mapping *tx_fifo_unmap;   /*!< To unmap the Tx FIFO */
  struct ef_hyperop_mapping *rx_fifo_unmap;   /*!< To unmap the Rx FIFO */
  atomic_t  destroying;               /*!< Prevent mutiple destruction */
  /* Resource tracking */
  struct ef_bend_limits quotas;       /*!< Limits on H/W & Dom0 resources */
  int num_mcasts;                     /*!< Number of multicasts used */
  int num_bufs;                       /*!< Number of NIC buffers used. */
  /* Hardware resources */
  int hw_type;                        /*!< The H/W type of associated NIC */
  int unaccelerated;   /*!< True if it will NOT be given H/W access even when available */
  int hw_state;        /*!< State of allocation */
  int (*accel_setup)(struct ef_bend *); /*!< How to set up the acceleration. */
  void (*accel_shutdown)(struct ef_bend *); /*!< And how to stop it. */
  struct efab_resource_s *ep_res;     /*!< Resource from PT endpoint allocator.
                                       * We hold on to the resources, but hand
                                       * over the H/W it describes to DomU */
  struct ef_iobufset_struct *iobufs;  /*!< I/O buffers for use by DomU */
  void *ep_vi;                        /*!< VI that goes with ep_res */
  struct ef_eventq_struct *ep_evq;    /*!< event Q we allocate for use by DomU */
  void *ep_evq_res;                   /*!< Resource associated with ep_evq */
  spinlock_t filter_lock;             /*!< Serialises access to filters */
  void **filter_res;                  /*!< Resources for filters DomU is using.*/
  struct ef_filter_spec *fspecs;
  unsigned long free_filters;         /*! Bitmap of which filters are free */
  __u32 filter_idx_mask;              /*! Used for index normalisation */
  int phys_port;                      /*!< The physical port to use */
  sh_msg_fifo2 to_domU;               /*!< Message FIFO */
  sh_msg_fifo2 from_domU;             /*!< Message FIFO */
  /* Network stack and slow path info. */
  sh_byte_fifo2 rcv_pkts;             /*!< Data FIFO,Rx slow path Dom0->DomU */
  sh_byte_fifo2 snd_pkts;             /*!< Data FIFO,Tx slow path DomU->DomO */
  ef_notify_t net_irq; /*!< Event channel dedicated to network packets. */
  __u8 mac[ETH_ALEN];  /*!< The MAC address the frontend goes by. */
  __u8 nicname[16];    /*!< Driver name of associated NIC */

  /* Stuff for the slow path TX network driver */
  struct net_device *netdev;
  struct net_device_stats *stats;
  struct proc_dir_entry *proc_entry;

  struct ef_bend *next_bend;
};

/*! \fn void ef_bend_mark_connected(struct ef_bend *bend)
 *
 * \brief Mark the backend as connected. On XEN this updates the xenbus
 * state. In fact this just requests that this is scheduled to happen
 * in a sleepable context.  Can be called in atomic 
 */
extern void ef_bend_mark_connected(struct ef_bend *bend);

/*! \fn void ef_bend_mark_dead(struct ef_bend *bend, int reason, const char *where);
 *
 * \brief Mark the backend as failed. On XEN this updates the xenbus state.
 *
 * \param reason An error number to describe the problem.  Passed to
 * xenbus_dev_fatal.
 * \param where A string to describe the problem.  Passed to
 * xenbus_dev_fatal.
 */
extern void ef_bend_mark_dead(struct ef_bend *bend, int reason, const char *where);

#endif

/*! \cidoxg_end */
