/**************************************************************************\
*//*! \file ef_bend_vnic.h backend <-> frontend communication

Copyright 2006 Solarflare Communications Inc,
               9501 Jeronimo Road, Suite 250,
               Irvine, CA 92618, USA

This program is free software; you can redistribute it and/or modify it under
the terms of the GNU General Public License version 2 as published by the Free
Software Foundation, incorporated herein by reference.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *//*
\**************************************************************************/

/* This file contains the Hypervisor- and OS- independent bits of the 
 * communication with the unprivileged VM (a.k.a. DomU if you're not fussy
 * about the Hypervisor-independence */

#include "ef_bend.h"
#include "ef_bend_vnic.h"
#include "ef_bend_fwd.h"
#include "ef_char_bend.h"
#include "ef_filter.h"
#include "ef_bend_netdev.h"

#if 0 /* Useful debugging tools */
static void show_mac(const char *txt, const __u8 *m)
{
  ci_log("%s %.2x:%.2x:%.2x:%.2x:%.2x:%.2x",
         txt, m[0], m[1], m[2], m[3], m[4], m[5]);
}


/* Print out a socket buffer */
void dump_pkt(struct sk_buff *skb)
{
  int i;
  char buf[80];
  char *buf_p = buf;
    
  ci_log("Len: %d", skb->len);
  show_mac("DST:", skb->data);
  show_mac("SRC:", skb->data + ETH_ALEN);
  ci_log("Payload:");
  for (i = ETH_HLEN; i < skb->len; i++) {
    buf_p += sprintf(buf_p, "%.2x ", skb->data[i]);
    if (((i - ETH_HLEN) & 15) == 15){
      ci_log("%s", buf);
      buf_p = buf;
      memset(buf, 0, 80);
    }
  }

  if(buf_p != buf)
    ci_log("%s", buf);
}
#endif

/* Handle an IRQ on the net channel - signifies there is a packet in
   the snd_pkts fifo */
static irqreturn_t pkt_from_vnic(int irq, void *context, struct pt_regs *unused)
{
  struct xenbus_device *dev = context;
  struct ef_bend *bend_data = (struct ef_bend *)dev->dev.driver_data;
  struct sk_buff *skb;
  void *skb_payload;

  BEND_VERB(ci_log("pkt (on irq %d) from device %s\n", irq, dev->nodename));

  while(sh_fifo2_not_empty(&bend_data->snd_pkts)){
    __u8 hi, lo;
    unsigned len;

    sh_fifo2_get(&bend_data->snd_pkts, &lo);
    sh_fifo2_get(&bend_data->snd_pkts, &hi);

    len = lo + (hi << 8);

    BEND_VERB(ci_log("got %d bytes from VNIC.", len));

    /* fill the packet */
    skb_payload = ef_bend_netdev_get_skb(len, &skb);
    memcpy(skb_payload, sh_fifo2_peekp(&bend_data->snd_pkts), len);

    /* Advance the read pointer to free up space in the fifo */
    sh_fifo2_rd_adv(&bend_data->snd_pkts, len);

    /* dump_pkt(skb); */

    /* pass to stack */
    ef_bend_netdev_recv(bend_data, skb);
  }

  return IRQ_HANDLED;
}


/* Send a HELLO to front end to start things off */
void ef_bend_vnic_hello(struct ef_bend * bend)
{
  ef_lock_state_t lock_state;
  struct ef_msg *msg = ef_msg_start_send(bend->shared_page,
                                         &bend->to_domU,
                                         &lock_state);
  /* The queue _cannot_ be full, we're the first users. */
  ci_assert(msg != NULL);
  ef_msg_init(msg, EF_MSG_HELLO);
  msg->u.hello.version = EF_VPROTO_VERSION;
  memcpy(msg->u.hello.mac, bend->mac, ETH_ALEN);
  ef_msg_complete_send_notify(bend->shared_page, &bend->to_domU,
                              &lock_state, bend->channel);
}

/* Send a local mac message to vnic */
static void ef_bend_vnic_localmac(struct ef_bend *bend, const void *mac, int type)
{
  ef_lock_state_t lock_state;
  struct ef_msg *msg;

  ci_assert(bend != NULL);
  ci_assert(mac != NULL);

  ci_log("Sending local mac message: " MAC_FMT, MAC_ARG((const char *)mac));  

  msg = ef_msg_start_send(bend->shared_page, &bend->to_domU, &lock_state);

  ci_assert(msg != NULL); /* Suggests queue is full, need to store the
                             mac and send a QUEUEFULL msg, then retry
                             later when the QUEUENOTFULL msg is
                             received in reply */

  ef_msg_init(msg, EF_MSG_LOCALMAC);
  msg->u.localmac.flags = type;
  memcpy(msg->u.localmac.mac, mac, ETH_ALEN);
  ef_msg_complete_send_notify(bend->shared_page, &bend->to_domU,
                              &lock_state, bend->channel);
}

/* Send an add local mac message to vnic */
void ef_bend_vnic_new_localmac(struct ef_bend *bend, const void *mac)
{
  ef_bend_vnic_localmac(bend, mac, EF_MSG_LOCALMAC_ADD);
}

/* Send a remove local mac message to vnic */
void ef_bend_vnic_old_localmac(struct ef_bend *bend, const void *mac)
{
  ef_bend_vnic_localmac(bend, mac, EF_MSG_LOCALMAC_REMOVE);
}


int ef_bend_buffer_map_request(struct ef_bend *bend, struct ef_msg *msg)
{
  int log2_pages, rc, i;
  efab_buffer_table_allocation bt_handle;
  struct vm_struct *area;
  /* Can only allocate in power of two */
  log2_pages = ci_log2_ge(msg->u.mapbufs.pages, 0);
  if(msg->u.mapbufs.pages != ci_pow2(log2_pages)){
    ci_log("%s: Can only allocate buffers in power of 2 sizes (not %d)",
           __FUNCTION__, msg->u.mapbufs.pages);
    rc = -EINVAL;
    goto err_out;
  }
  
  /* Sanity.  Assumes EF_MSG_MAX_PAGE_REQ is same for both
     directions/domains */
  if(msg->u.mapbufs.pages > EF_MSG_MAX_PAGE_REQ){
    ci_log("%s: too many pages in a single message: %d %d", 
           __FUNCTION__, msg->u.mapbufs.pages, EF_MSG_MAX_PAGE_REQ);
    rc = -EINVAL;
    goto err_out;
  }
  
  if((rc = ef_bend_buffer_table_alloc(log2_pages, &bt_handle)) < 0){
    goto err_out;
  }
  
  for(i = 0; i < msg->u.mapbufs.pages; i++){
    area = xenbus_map_ring_valloc(bend->hdev_data, 
                                  msg->u.mapbufs.reqs[i].grant);
    
    if (IS_ERR(area)) {
      /* This is bad, as we don't undo the maps or buffer table entries */
      rc = PTR_ERR(area);
      goto err_out;
    }

    if((rc = ef_bend_buffer_table_set(bend, msg->u.mapbufs.mfns[i], i,
                                      &bt_handle)) < 0){
      /* This is bad, as we don't undo the maps or buffer table entries */
      goto err_out;
    }
  }

  msg->id |= EF_MSG_REPLY;
  
  msg->u.mapbufs.buf = EFAB_BUFFER_ADDR(bt_handle.base, 0);

  return 0;

 err_out:
  ci_log("%s: err_out", __FUNCTION__);
  msg->id |= EF_MSG_ERROR | EF_MSG_REPLY;
  return rc;
}

/* Wrapper round the buffer allocation. */
int ef_bend_buffer_request(struct ef_bend *bend, struct ef_msg *msg)
{
  int pages = msg->u.bufs.pages;
  void **ptrs = kzalloc(sizeof(void *) * pages, GFP_KERNEL);

  int rc;
  int i;

  /* Sanity */
  if(pages > EF_MSG_MAX_PAGE_REQ){
    ci_log("%s: too many pages in a single message: %d %d", 
           __FUNCTION__, pages, EF_MSG_MAX_PAGE_REQ);
    rc = -EINVAL;
    goto out;
  }

  /* Allocate an iobufset */
  rc = ef_get_bufs(bend, &pages, ptrs, &msg->u.bufs.buf);

  if (rc == 0) {
    msg->id |= EF_MSG_REPLY;
    ci_log("About to set up buffer grants.\n");
    for (i = 0; i < pages; i++) {
      /* Set up buffer grants */
      msg->u.bufs.reqs[i].grant = 
        ef_hyperop_pass_mem_page(bend->hdev_data, "", 
                                 page_to_pfn(virt_to_page(ptrs[i])),
                                 bend->far_end);
    }
    msg->u.bufs.pages = pages;
  } else {
    msg->id |=  EF_MSG_ERROR | EF_MSG_REPLY;
  }

 out:
  kfree(ptrs);
  return rc;
}


inline void ef_bend_set_queue_not_full(struct ef_bend *bend)
{
  if(!ci_bit_test_and_set(&bend->shared_page->aflags, 
                          MSG_AFLAGS_QUEUEUNOTFULL_B)){
    /* Clear the QUEUEFULL bit */
    ci_assert(ci_bit_test(&bend->shared_page->aflags, MSG_AFLAGS_QUEUEUFULL_B));
    ci_bit_clear(&bend->shared_page->aflags, MSG_AFLAGS_QUEUEUFULL_B);
    ef_hyperop_remote_irq(bend->channel);
  } else
    ci_log("queue not full bit already set, not signalling");
}

inline void ef_bend_set_queue_full(struct ef_bend *bend)
{
  if(!ci_bit_test_and_set(&bend->shared_page->aflags, MSG_AFLAGS_QUEUE0FULL_B))
    ef_hyperop_remote_irq(bend->channel);
  else
    ci_log("queue full bit already set, not signalling");
}


static 
int ef_bend_lock_msg_replies(struct ef_msg *msg, sh_msg_fifo2 *queue, 
                             unsigned *flags)
{
  int reply_slots;
  switch(msg->id) {
  case EF_MSG_GETBUF:
  case EF_MSG_MAPBUF:
    reply_slots = 1;
    break;
  case EF_MSG_RXMEM:
  case EF_MSG_TXMEM:
    reply_slots = 0;
    break;
  default:
    if(msg->id & EF_MSG_REPLY)
      reply_slots = 0;
    else{
      ci_log("%s: Unknown number of reply slots required", __FUNCTION__);
      /* Assume zero for now */
      reply_slots = 0;
    }
    break;
  }

  if(reply_slots){
    ef_msg_lock_queue(queue, flags);
    if(!ef_msg_check_space(queue, reply_slots)){
      ef_msg_unlock_queue(queue, flags);
      return -ENOSPC;
    }
  }

  return reply_slots;
}

/*  Demultiplex an IRQ from the frontend driver.  */
void ef_bend_vnic_msghandler(struct ef_bend *bend)
{
  struct ef_msg msg;
  int err, cookie, reply_slots;
  unsigned flags;

  while((err = ef_msg_peek(bend->shared_page, &bend->from_domU, &msg, &cookie))
        == 0){
    
    if((reply_slots = ef_bend_lock_msg_replies(&msg, &bend->to_domU, &flags))
       >= 0){
      switch(msg.id) {
      case EF_MSG_REPLY | EF_MSG_HELLO:
        /* Reply to a HELLO; mark ourselves as connected */
        ci_log("got Hello reply, version %.8x", msg.u.hello.version);
        ef_bend_mark_connected(bend);
        /* Set up the hardware visible to the other end */
        err = bend->accel_setup(bend);
        if (err) {
          /* This is fatal: even the software fallback hasn't been set up. */
          ef_bend_mark_dead(bend, err, "Could not setup H/W");
        } else
          bend->hw_state = EF_RES_ALLOC;
        break;
      case EF_MSG_MAPBUF:
        ci_log("Got mapped buffers request");
        /* Front end wants a buffer table entry for the supplied pages */
        err = ef_bend_buffer_map_request(bend, &msg);
        ef_msg_send_notify_locked(bend->shared_page, bend->channel, 
                                   &bend->to_domU, &msg);
        break;
      case EF_MSG_GETBUF:
        /* Front end wants a buffer table entry, better try and get it one */
        err = ef_bend_buffer_request(bend, &msg);
        ef_msg_send_notify_locked(bend->shared_page, bend->channel, 
                                  &bend->to_domU, &msg);
        break;
      case EF_MSG_RXMEM:
        /* TODO - this assumes it hasn't received one of these before */
        
        /* Allocate shared pages for. rx msg.u.rxmem.pages is actually
         * one smaller than the number of grants provided in the
         * grants array - we map the first page twice to provide a
         * partial virtual ring buffer */
        sh_fifo2_init(&bend->rcv_pkts, msg.u.pktmem.pages * PAGE_SIZE - 1,
                      &bend->shared_page->rcv_pkts_rd, 
                      &bend->shared_page->rcv_pkts_wr);

        /* Map the pages described by the message, mapping the first page in 
         * again at the end to implement the virtual ring buffer. */
        bend->rcv_pkts.fifo = ef_hyperop_map_pktmem(bend->hdev_data, &msg,
                                                    &bend->rx_fifo_unmap);

        BEND_VERB(ci_log("Set pkt FIFO mask to %x\n", bend->rcv_pkts.fifo_mask));

        if (bend->rcv_pkts.fifo != NULL) {
          /* Activate our entry in the forward hash table */
          err = ef_bend_fwd_set_context(bend->mac, bend);
          /* This has no business failing since we already had a table 
           * entry. */
          ci_assert(err == 0);
        }
        else {
          ef_bend_mark_dead(bend, -EINVAL, "FATAL: fail to map rx page grants.");
        }
        break;
      case EF_MSG_TXMEM:
        ci_log("%s: got EF_MSG_TXMEM", __FUNCTION__);
        /* TODO - this assumes it hasn't received one of these before */

        /* Similar to RXMEM, but this time we've got a region for the
           front-end driver to send us packets (tx slow path)
           msg.u.pktmem.pages is actually one smaller than the number of
           grants provided in the grants array - we map the first page
           twice to provide a partial virtual ring buffer */
        sh_fifo2_init(&bend->snd_pkts, msg.u.pktmem.pages * PAGE_SIZE - 1,
                      &bend->shared_page->snd_pkts_rd, 
                      &bend->shared_page->snd_pkts_wr);
      
        /* Map the pages described by the message, mapping the first page in 
         * again at the end to implement the virtual ring buffer. */
        bend->snd_pkts.fifo = ef_hyperop_map_pktmem(bend->hdev_data, &msg,
                                                    &bend->tx_fifo_unmap);
        if (bend->snd_pkts.fifo == NULL) {
          ef_bend_mark_dead(bend, -EINVAL, "FATAL: fail to map tx page grants.");
        }

        /* Now the fifo is set up we can bind the irq handler */
        err = ef_hyperop_set_evhandler(bend->net_irq, pkt_from_vnic,
                                       SA_SAMPLE_RANDOM, "ef1net", 
                                       bend->hdev_data);
        if(err < 0)
          ci_log("failed to bind net channel to IRQ: %d\n", err);
        break;
      default:
        ci_log("Huh? Message code is %x", msg.id);
        break;
      }

      if(reply_slots){
        /* The queue is still locked, release it now we've finished
           sending replies */
        ef_msg_unlock_queue(&bend->from_domU, &flags);
      }
      
      /* Finally, remove the message from the queue. */
      err = ef_msg_recv_next(bend->shared_page, &bend->from_domU, cookie);
      /* Can assert this as it should be the same error we got from
         peek */
      ci_assert_equal(err, 0);
    }
    else{
      ci_log("%s: no space for reply, not processing incoming message",
             __FUNCTION__);
      /* Couldn't get the reply slots needed */
      /* Don't process the incoming message until possible to reply */
      ef_bend_set_queue_full(bend);
      /* Give up for now */
      break;
    }
  }

  if((bend->shared_page->aflags & MSG_AFLAGS_TO_DOM0_MASK) != 0){
    if(bend->shared_page->aflags & MSG_AFLAGS_QUEUE0NOTFULL){
      /* We've been told there may now be space.  Of course, the above
         bits of this function may have already used that space (and
         notified that it's full again) */
      ci_bit_clear(&bend->shared_page->aflags, MSG_AFLAGS_QUEUE0NOTFULL_B);

      /* TODO see if there are any mac address changes waiting to be
         sent */
    }

    if(bend->shared_page->aflags & MSG_AFLAGS_QUEUEUFULL){
      /* There will be space now if we can make any.  TODO make this
         more intelligent, so that separate interrupts aren't used for
         sending replies (if present) and this message */
      ef_bend_set_queue_not_full(bend);
    }
  }

  /* Ignore ENOENT as that is used to break out of the loop */
  if(err != -ENOENT)
    ci_log("%s: Error %d", __FUNCTION__, err);

  return;
}
