/**************************************************************************\
*//*! \file ef_cuckoo_hash.h Cuckoo hash table

Copyright 2006 Solarflare

This program is free software; you can redistribute it and/or modify it under
the terms of the GNU General Public License version 2 as published by the Free
Software Foundation, incorporated herein by reference.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program; if not, write to the Free Software
Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA

 *//*
\**************************************************************************/

/* A cuckoo hash table consists of two sub tables.  Each entry can
   hash to a position in each table.  If, on entry, its position is
   found to be occupied, the existing element is moved to it's other
   location.  This recurses until success or a loop is found.  If a
   loop is found the table is rehashed.

   See http://www.it-c.dk/people/pagh/papers/cuckoo-jour.pdf
*/

/* TODO make this less specialised to the problem of storing mac
   addresses */

#ifndef EF_CUCKOO_HASH_H
#define EF_CUCKOO_HASH_H

#define EF_CUCKOO_HASH_KEY_BITS (48)
#define EF_CUCKOO_HASH_KEY_BYTES (6)

/*! Type used for hash table keys */
typedef ci_uint64 ef_cuckoo_hash_key;

/*! Type used for the values stored in the hash table */
typedef int ef_cuckoo_hash_value;

/*! Type used for the hash used to index the table */
typedef ci_uint32 ef_cuckoo_hash;

/*! How long to spend displacing values when adding before giving up
  and rehashing */
#define EF_CUCKOO_HASH_MAX_LOOP (hashtab->length)

/*! State of hash table entry */
typedef enum {
  EF_CUCKOO_HASH_STATE_VACANT = 0,
  EF_CUCKOO_HASH_STATE_OCCUPIED 
} ef_cuckoo_hash_state;

/*! An entry in the hash table */
typedef struct {
  ef_cuckoo_hash_state state;
  ef_cuckoo_hash_key key;
  ef_cuckoo_hash_value value;
} ef_cuckoo_hash_entry;

/*! A cuckoo hash table */
typedef struct {
  unsigned length; /*!< The length of each table (NB. there are two
                     tables of this length */
  unsigned length_bits; /*!< The length of each table in bits */
  unsigned entries; /*!< The number of entries currently stored in the table */

  /* parameter of hash functions */
  ef_cuckoo_hash_key a0; /*!< The "a" parameter of the first hash function */
  ef_cuckoo_hash_key a1; /*!< The "a" parameter of the second hash function */

  ef_cuckoo_hash_entry *table0; /*!< The first table */
  ef_cuckoo_hash_entry *table1; /*!< The second table */

} ef_cuckoo_hash_table;

/*! Initialise the cuckoo has table 
 *
 * \param hashtab A pointer to an unitialised hash table structure
 * \param length_bits The number of elements in each table equals
 * 2**length_bits
 *
 * \return 0 on success, -ENOMEM if it couldn't allocate the tables
 */
int ef_cuckoo_hash_init(ef_cuckoo_hash_table *hashtab, unsigned length_bits);


/*! Destroy a hash table
 *
 * \param hashtab A hash table that has previously been passed to a
 * successful call of ef_cuckoo_hash_init()
 */
void ef_cuckoo_hash_destroy(ef_cuckoo_hash_table *hashtab);


/*! Lookup an entry in the hash table 
 *
 * \param hashtab The hash table in which to look.
 * \param key Pointer to a mac address to use as the key
 * \param value On exit set to the value stored if key was present
 *
 * \return 0 if not present in the table, non-zero if it is (and value
 * is set accordingly)
 */
int ef_cuckoo_hash_lookup(ef_cuckoo_hash_table *hashtab, const char *key,
                          ef_cuckoo_hash_value *value);

/*! Add an entry to the hash table 
 *
 * \param hashtab The hash table to add the entry to
 * \param key Pointer to a mac address to use as a key
 * \param value The value to store 
 * \param can_rehash Flag to allow the add function to rehash the
 * table if necessary
 *
 * \return 0 on success, non-zero on failure.  -EBUSY means the key is
 * already present in the table. -ENOSPC means it just couldn't find
 * anywhere to put it - this is bad and probably means an entry has
 * been dropped on the floor (but the entry you just tried to add may
 * now be included)
 */
int ef_cuckoo_hash_add(ef_cuckoo_hash_table *hashtab,
                       const char *key, ef_cuckoo_hash_value value,
                       int can_rehash);

/*! Remove an entry from the table 
 *
 * \param hashtab The hash table to remove the entry from
 * \param key The key that was used to previously add the entry
 *
 * \return 0 on success, -EINVAL if the entry couldn't be found 
 */
int ef_cuckoo_hash_remove(ef_cuckoo_hash_table *hashtab, const char *key);

#endif /* EF_CUCKOO_HASH_H */
