(* 
    Messages for OCaml XenStore Daemon.
    Copyright (C) 2008 Patrick Colp University of British Columbia

    This program is free software; you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation; either version 2 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program; if not, write to the Free Software
    Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
*)

let header_size = 16;

(* XenStore message types *)
type xs_message_type =
  | XS_DEBUG
  | XS_DIRECTORY
  | XS_READ
  | XS_GET_PERMS
  | XS_WATCH
  | XS_UNWATCH
  | XS_TRANSACTION_START
  | XS_TRANSACTION_END
  | XS_INTRODUCE
  | XS_RELEASE
  | XS_GET_DOMAIN_PATH
  | XS_WRITE
  | XS_MKDIR
  | XS_RM
  | XS_SET_PERMS
  | XS_WATCH_EVENT
  | XS_ERROR
  | XS_IS_DOMAIN_INTRODUCED
  | XS_RESUME
  | XS_SET_TARGET
  | XS_UNKNOWN

(* Convert a message type to an int32 *)
let xs_message_type_to_int32 message_type =
  match message_type with
  | XS_DEBUG -> 0l
  | XS_DIRECTORY -> 1l
  | XS_READ -> 2l
  | XS_GET_PERMS -> 3l
  | XS_WATCH -> 4l
  | XS_UNWATCH -> 5l
  | XS_TRANSACTION_START -> 6l
  | XS_TRANSACTION_END -> 7l
  | XS_INTRODUCE -> 8l
  | XS_RELEASE -> 9l
  | XS_GET_DOMAIN_PATH -> 10l
  | XS_WRITE -> 11l
  | XS_MKDIR -> 12l
  | XS_RM -> 13l
  | XS_SET_PERMS -> 14l
  | XS_WATCH_EVENT -> 15l
  | XS_ERROR -> 16l
  | XS_IS_DOMAIN_INTRODUCED -> 17l
  | XS_RESUME -> 18l
  | XS_SET_TARGET -> 19l
  | XS_UNKNOWN -> - 1l

(* Convert an int32 to a message type *)
let int32_to_message_type xs_message_type =
  match xs_message_type with
  | 0l -> XS_DEBUG
  | 1l -> XS_DIRECTORY
  | 2l -> XS_READ
  | 3l -> XS_GET_PERMS
  | 4l -> XS_WATCH
  | 5l -> XS_UNWATCH
  | 6l -> XS_TRANSACTION_START
  | 7l -> XS_TRANSACTION_END
  | 8l -> XS_INTRODUCE
  | 9l -> XS_RELEASE
  | 10l -> XS_GET_DOMAIN_PATH
  | 11l -> XS_WRITE
  | 12l -> XS_MKDIR
  | 13l -> XS_RM
  | 14l -> XS_SET_PERMS
  | 15l -> XS_WATCH_EVENT
  | 16l -> XS_ERROR
  | 17l -> XS_IS_DOMAIN_INTRODUCED
  | 18l -> XS_RESUME
  | 19l -> XS_SET_TARGET
  | _ -> XS_UNKNOWN

(* Return string representation of a message type *)
let message_type_to_string message_type =
  match message_type with
  | XS_DEBUG -> "DEBUG"
  | XS_DIRECTORY -> "DIRECTORY"
  | XS_READ -> "READ"
  | XS_GET_PERMS -> "GET_PERMS"
  | XS_WATCH -> "WATCH"
  | XS_UNWATCH -> "UNWATCH"
  | XS_TRANSACTION_START -> "TRANSACTION_START"
  | XS_TRANSACTION_END -> "TRANSACTION_END"
  | XS_INTRODUCE -> "INTRODUCE"
  | XS_RELEASE -> "RELEASE"
  | XS_GET_DOMAIN_PATH -> "GET_DOMAIN_PATH"
  | XS_WRITE -> "WRITE"
  | XS_MKDIR -> "MKDIR"
  | XS_RM -> "RM"
  | XS_SET_PERMS -> "SET_PERMS"
  | XS_WATCH_EVENT -> "WATCH_EVENT"
  | XS_ERROR -> "ERROR"
  | XS_IS_DOMAIN_INTRODUCED -> "IS_DOMAIN_INTRODUCED"
  | XS_RESUME -> "RESUME"
  | XS_SET_TARGET -> "SET_TARGET"
  | XS_UNKNOWN -> "UNKNOWN"

(* Message header *)
type header =
  {
    message_type : xs_message_type;
    transaction_id : int32;
    request_id : int32;
    length : int
  }

(* Message *)
type message =
  {
    header : header;
    payload : string
  }

(* Make a message *)
let make message_type transaction_id request_id payload =
  {
    header =
      {
        message_type = message_type;
        transaction_id = transaction_id;
        request_id = request_id;
        length = (String.length payload)
      };
    payload = payload
  }

(* Null message *)
let null_message = make XS_UNKNOWN 0l 0l Constants.null_string

(* ACK message *)
let ack message =
  make message.header.message_type message.header.transaction_id message.header.request_id (Utils.null_terminate "OK")

(* Error message *)
let error message error =
  make XS_ERROR message.header.transaction_id message.header.request_id (Utils.null_terminate (Constants.error_message error))

(* Event message *)
let event payload =
  make XS_WATCH_EVENT 0l 0l payload

(* Reply message *)
let reply message payload =
  make message.header.message_type message.header.transaction_id message.header.request_id payload

(* Deserialise a message header from a string *)(* Null message *)
let deserialise_header buffer =
  {
    message_type = int32_to_message_type (Utils.bytes_to_int32 (String.sub buffer 0 4));
    transaction_id = Utils.bytes_to_int32 (String.sub buffer 8 4);
    request_id = Utils.bytes_to_int32 (String.sub buffer 4 4);
    length = Utils.bytes_to_int (String.sub buffer 12 4)
  }

(* Serialise a message header to a string *)
let serialise_header header =
  let message_type = Utils.int32_to_bytes (xs_message_type_to_int32 header.message_type)
  and transaction_id = Utils.int32_to_bytes header.transaction_id
  and request_id = Utils.int32_to_bytes header.request_id
  and length = Utils.int_to_bytes header.length in
  message_type ^ request_id ^ transaction_id ^ length
