#!/bin/bash
# This is a build script to create a Debian based Linux kernel 4.9 and filesystem based on debian 9.2
# https://eewiki.net/display/linuxonarm/BeagleBoard-X15

# version 0.1.0 
#  Dec 2017



# requires package devscripts to be installed

DEBIAN_FS=debian-9.3-minimal-armhf-2017-12-09
KERNEL_TAG=4.9.74-ti-rt-r90
UBOOT_TAG=2017.01


# Are we booting kernel and rootfs over network
NETWORK_BOOT=0

BUILD_SRC_DIR=`pwd`

if [ ${NETWORK_BOOT} -eq 1 ]
then 
   # set SERVER_IP to put into uboot
  SERVER_IP=192.168.1.231

fi

if [ ! -e toolchain-downloaded ]
then

   wget -c https://releases.linaro.org/components/toolchain/binaries/6.4-2017.08/arm-linux-gnueabihf/gcc-linaro-6.4.1-2017.08-x86_64_arm-linux-gnueabihf.tar.xz
   tar xf gcc-linaro-6.4.1-2017.08-x86_64_arm-linux-gnueabihf.tar.xz
   touch toolchain-downloaded
fi
export CC=`pwd`/gcc-linaro-6.4.1-2017.08-x86_64_arm-linux-gnueabihf/bin/arm-linux-gnueabihf-


if [ ! -e fs-downloaded ]
then
   if [ ! -e fs ]
   then
      mkdir fs
   fi

   cd fs || exit
   wget -c https://rcn-ee.com/rootfs/eewiki/minfs/${DEBIAN_FS}.tar.xz
   tar xf ${DEBIAN_FS}.tar.xz

   cd ${DEBIAN_FS}
   mkdir -p rootfs
   
  
   cd rootfs
   FS=`pwd`

   # extract FS
   sudo tar xf ../armhf-rootfs-debian-stretch.tar . 
   cd ..
   # and then change ownership to root
   sudo chown root:root rootfs/
   sudo chmod 755 rootfs/
   cd ..

   touch ../fs-downloaded
   cd ..
else
   # just set FS
   FS=`pwd`/fs/${DEBIAN_FS}/rootfs
   cd fs/${DEBIAN_FS} || exit
   sudo rm -r rootfs/ 
   mkdir -p rootfs
   cd rootfs
   sudo tar xf ../armhf-rootfs-debian-stretch.tar . 
   cd ..
   # and then change ownership to root
   sudo chown root:root rootfs/
   sudo chmod 755 rootfs/
   cd ../..
fi



if [ ! -e uboot-downloaded ]
then

    git clone https://github.com/u-boot/u-boot
    cd u-boot/

     # update to latest tag 
    git checkout v${UBOOT_TAG} -b tmp || exit

    
    # and patch it 
    git pull --no-edit git://git.ti.com/ti-u-boot/ti-u-boot.git ti-u-boot-${UBOOT_TAG}
    git checkout 590c7d7fe15d06a4d708403d3234bcd01cd039e1
    
    wget -c https://rcn-ee.com/repos/git/u-boot-patches/ti-${UBOOT_TAG}/0001-beagle_x15-uEnv.txt-bootz-n-fixes.patch || exit
    git am 0001-beagle_x15-uEnv.txt-bootz-n-fixes.patch || exit


  if [ ${NETWORK_BOOT} -eq 1 ]
  then 

     # modify patch to handle actual NFS on server. we point uboot to a generic targetNFS directory
     # in the sdk as this is the name exported in /etc/exports on this build machine
     sed "s:NFS:${FS}:g" ../patches/u-boot/clean-0001-set-netboot-as-default-for-uboot.patch > ../patches/u-boot/0001-set-netboot-as-default-for-uboot.patch

     # and set the serverip for this build
     sed "s:IP_ADDR:${SERVER_IP}:g" ../patches/u-boot/clean-0002-allow-serverip-to-be-set-by-build.patch > ../patches/u-boot/0002-allow-serverip-to-be-set-by-build.patch 

     # then apply the modified patches  
     git am ../patches/u-boot/0001-set-netboot-as-default-for-uboot.patch || exit

     # This patch sets serverip and then in netboot resets it after dhcp in case that dhcp has set it with the router address
     git am ../patches/u-boot/0002-allow-serverip-to-be-set-by-build.patch || exit
     
     # for booting into xen set envboot as default so that boot.scr is read
     git am ../patches/u-boot/0003-set-envboot-as-default-for-xen.patch || exit
     
  else
     # for booting into xen set envboot as default so that boot.scr is read
    git am ../patches/u-boot/0001-for-xen-build-set-envboot-as-default-mmc.patch || exit

  fi

  # as u-boot is rarely rebuilt just do it once after download
  make ARCH=arm CROSS_COMPILE=${CC} distclean || exit   

  touch ../uboot-downloaded
  cd ..
fi

cd u-boot/
make ARCH=arm CROSS_COMPILE=${CC} am57xx_evm_defconfig || exit
make ARCH=arm CROSS_COMPILE=${CC} || exit
cd ..


#Download kernel and build it clean
#part of the process is to download gcc 4.9 and so do this before uboot
if [ ! -e kernel-built ]
then
   if [ ! -e kernel-git-downloaded ]
   then
      sudo rm -r ti-linux-kernel-dev
      git clone https://github.com/RobertCNelson/ti-linux-kernel-dev.git
      touch kernel-git-downloaded 
   fi 
    
  
   if [ ! -e kernel-patched ]
   then
     cd ti-linux-kernel-dev/
     git checkout ${KERNEL_TAG} 
 

     echo "CORES=8" >> system.sh.sample
 
     # add patches 
     git am ../patches/kernel/0001-Add-XEN-support-to-kernel.patch || exit
     
     git am ../patches/kernel/0002-change-CMA_SIZE-to-24M-to-match-TI-SDK.patch || exit

     # add patch to directory of patches to be applied by the build 
     cp ../patches/kernel/0003-merge-SDK-cmem-allocation.patch  patches/soc/ti/x15
     git am ../patches/rcn-build/0001-add-xen-rt-kernel-patch.patch || exit
     cp ../patches/kernel/0002-rt-build-xen-for-RT-kernel.patch  patches/rt
     
     
     if [ ${NETWORK_BOOT} -eq 1 ]
     then 

        # populate DTS file patch for with correct NFS info      
        NFS_SETUP="${SERVER_IP}:${FS},nolock"
        sed "s#NFS_ROOT#${NFS_SETUP}#g" ../patches/kernel/clean-0004-add-xen-configuration-to-DTS-file.patch > ../patches/kernel/0004-rt-add-xen-configuration-to-DTS-file.patch
        cp ../patches/kernel/0004-rt-add-xen-configuration-to-DTS-file.patch patches/rt
     else
       cp ../patches/kernel/0004-add-xen-configuration-to-DTS-file-mmc-boot.patch patches/rt/0004-rt-add-xen-configuration-to-DTS-file.patch
     
     fi
     
     
     git am ../patches/rcn-build/0002-add-DTS-patch-for-Xen.patch || exit
     
     
     cp ../patches/kernel/0005-change-chosen-to-be-uart3-in-dts.patch patches/rt
     git am ../patches/rcn-build/0003-add-uart3-change-to-chosen.patch  || exit
     
     cp ../patches/kernel/0006-add-psci-support-to-device-tree.patch patches/rt
     git am ../patches/rcn-build/0004-add-psci-patch-to-dt.patch || exit
     
     cp ../patches/kernel/0007-remove-CPU1-to-leave-single-core-in-DT.patch patches/rt
     git am ../patches/rcn-build/0005-change-to-single-A15-core.patch || exit
     
     cp ../patches/kernel/0008-set-armv7-timer-to-6-14MHz-rate-in-DT.patch patches/rt || exit
     git am ../patches/rcn-build/0006-add-armv7-timer-frequency-to-DT.patch || exit
     
     
     
     git am ../patches/kernel/0009-enable-ARM_LPAE-in-kernel-config.patch || exit
     git am ../patches/kernel/0010-enable-debug-symbols-in-kernel.patch || exit
 
     cp ../patches/kernel/0011-change-command-line-values-to-a-single-string-in-DT.patch patches/rt || exit
     git am ../patches/rcn-build/0007-add-0011-change-command-line-values.patch || exit
     
     cp ../patches/kernel/0012-ensure-xens-fake-interrupt-controller-node-is-marked.patch patches/rt || exit
     git am ../patches/rcn-build/0008-add-handler-for-xens-virtual-interrupt-controller.patch || exit
     
    
     cp ../patches/kernel/0013-temp-remove-cma-pools-for-co-processors-as-xen-does-.patch patches/rt || exit
     git am ../patches/rcn-build/0009-add-0013-to-remove-co-pro-cma-pools.patch || exit
     
     
     cp ../patches/kernel/0014-explicitly-reference-all-hw-used-by-the-kernel-in-DT.patch patches/rt || exit
     git am ../patches/rcn-build/0010-add-0014-explicit-DT-references.patch || exit
    
     cd ..
     touch kernel-patched
   fi
   
   
   cd ti-linux-kernel-dev/
   ./build_kernel.sh || exit
  # ./build_kernel_partial.sh || exit
  cd .. 
   touch kernel-built

fi
  
  

# pick up kernel_version from kernel build directory
cd ti-linux-kernel-dev
export kernel_version=`cat kernel_version`
cd ..
echo "kernel_version:${kernel_version}"


# now that kernel is built copy the zImage and modules to rootfs
sudo mkdir -p ${FS}/boot/dtbs/${kernel_version}/
  
  
# take base uEnv.txt from package and then add specifics to it 
# This can't be done in this script via echo as we do not want the 
# variables like ${loadaddr} to be evaluated by the shell now. We 
# want them to be evaluated by u-boot at runtime
#sudo cp patches/uEnv.txt ${FS}/boot/uEnv.txt
  
# want this expression evaluated now, not at boot
sudo sh -c "echo 'uname_r=${kernel_version}' >> ${FS}/boot/uEnv.txt"



if [ ${NETWORK_BOOT} -eq 1 ]
then 
   # copy kernel and dtb to tftp
   sudo cp -v ti-linux-kernel-dev/deploy/${kernel_version}.zImage /tftpboot/zImage || exit
   sudo tar -xf ti-linux-kernel-dev/deploy/${kernel_version}-dtbs.tar.gz -C /tftpboot/ || exit
fi

# for xen still need dtbs file in FS for xen to read
sudo tar xf ti-linux-kernel-dev/deploy/${kernel_version}-dtbs.tar.gz -C ${FS}/boot/dtbs/${kernel_version}/ || exit
sudo cp ti-linux-kernel-dev/deploy/${kernel_version}.zImage ${FS}/boot/zImage || exit

sudo tar xf ti-linux-kernel-dev/deploy/${kernel_version}-modules.tar.gz -C ${FS}/ || exit
   
# set up file system table
sudo sh -c "echo '/dev/mmcblk0p1  /  auto  errors=remount-ro  0  1' >> ${FS}/etc/fstab"



if [ ! -e xen-downloaded ]
then
  # based on https://wiki.xen.org/wiki/Xen_ARM_with_Virtualization_Extensions/OMAP5432_uEVM

  git clone git://xenbits.xen.org/xen.git
  
  cd xen
  git checkout RELEASE-4.8.3
  cd ..
  
  touch xen-downloaded
fi


cd xen
if [ ! -e ../xen-patched ]
then
    git am ../patches/xen/0001-for-AM572x-need-to-flush-TLBs-before-enabling-MMU.patch || exit
    git am ../patches/xen/0002-skip-test-for-interrupt-node-as-AM572x-has-two-optio.patch || exit
    git am ../patches/xen/0003-add-PRCM_MPU-to-memory-translation-for-AM572x.patch || exit
    git am ../patches/xen/0004-add-an-interrupt-parent-property-to-xens-own-interru.patch || exit
    # implementation of one solution from https://lists.xenproject.org/archives/html/xen-devel/2015-07/msg05345.html
    git am ../patches/xen/0005-DIRTY-HACK-allow-Dom0-to-access-secure-registers-dir.patch || exit

    touch ../xen-patched 
fi




 # ./configure --enable-debug --enable-debugger --enable-verbose


  
  # x15 also uses UART3 so reuse omap5432 definition
 # make clean XEN_TARGET_ARCH=arm32 CROSS_COMPILE=${CC} CONFIG_DEBUG=y CONFIG_DEVICE_TREE_DEBUG=y CONFIG_EARLY_PRINTK=omap5432
 #  make dist-xen XEN_TARGET_ARCH=arm32 CROSS_COMPILE=${CC} CONFIG_DEBUG=y CONFIG_DEVICE_TREE_DEBUG=y CONFIG_EARLY_PRINTK=omap5432
  make xen XEN_TARGET_ARCH=arm32 CROSS_COMPILE=${CC} CONFIG_DEBUG=y CONFIG_DEVICE_TREE_DEBUG=y CONFIG_EARLY_PRINTK=omap5432 || exit

  mkimage -A arm -T kernel -a 0x80200000 -e 0x80200000 -C none -d "xen/xen" xen-uImage

  cd ..



######################################################################
# Now create a boot.scr file for uboot to use to configure xen_addr_r
# 

cd xen

 sh -c "echo '
setenv dtb_addr_r 0x825f0000
setenv xen_addr_r 0x90000000
setenv kernel_addr_r 0xa0000000
ext4load mmc 0:1 \$dtb_addr_r boot/dtbs/${kernel_version}/am57xx-beagle-x15-revc.dtb
ext4load mmc 0:1 \$xen_addr_r xen-uImage
ext4load mmc 0:1 \$kernel_addr_r boot/zImage
bootm \$xen_addr_r - \$dtb_addr_r'  > boot.xen" || exit

mkimage -A arm -T script -d boot.xen boot.scr

cd ..





cd ${BUILD_SRC_DIR}
mkdir -p mmc-package

sudo cp -v ./u-boot/MLO mmc-package/
sudo cp -v ./u-boot/u-boot.img mmc-package/

sudo cp -v ./xen/boot.scr ${FS}/
sudo cp -v ./xen/xen-uImage ${FS}/ 




# now tar up the rootfs
cd ${FS}

   if [ ${NETWORK_BOOT} -eq 0 ]
   then 
       # create full tar file for SD filesystem
       sudo tar -cf ../../tar-rootfs.tar * || exit

   else
       # create a dummy filesystem so that create-sdcard.sh will quickly created a boot partition
       sudo tar -c --no-recursion -f ../../tar-rootfs.tar * || exit

       sudo tar -rf  ../../tar-rootfs.tar boot/zImage || exit
       sudo tar -rf  ../../tar-rootfs.tar boot/dtbs/4.9.74-ti-rt-r90/am57xx-beagle-x15-revc.dtb || exit

   fi    

 
   
cd ../../..



#now tar up package to programme, the file mmc-package.tar.gz is file used by the programme-sd.sh script
cd fs || exit
sudo tar -cf ../mmc-package/mmc-package.tar tar-rootfs.tar
cd ../mmc-package
sudo tar -rf mmc-package.tar u-boot.img MLO  
cd ..

echo "Build Successful"

